﻿using Hims.Api.Models;
using Hims.Api.Utilities;
using Hims.Domain.Services;
using Hims.Shared.DataFilters;
using Hims.Shared.EntityModels;
using Hims.Shared.Library.Enums;
using Hims.Shared.UserModels;
using Microsoft.AspNetCore.Mvc;
using System.Threading.Tasks;
using System;
using CorePush.Apple;
using System.Security.AccessControl;
using Hims.Shared.UserModels.Common;
using Microsoft.AspNetCore.Authorization;
using DocumentFormat.OpenXml.Drawing.Charts;
using Hims.Domain.Entities;
using Npgsql;

namespace Hims.Api.Controllers
{
    /// <inheritdoc />
    [Authorize]
    [Route("api/reasons")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class ReasonsController : BaseController
    {
        /// <summary>
        /// The reason services.
        /// </summary>
        private readonly IReasonsService reasonsService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public ReasonsController(IReasonsService reasonsService, IAuditLogService auditLogService)
        {
            this.reasonsService = reasonsService;
            this.auditLogServices = auditLogService;
        }

        /// <summary>
        /// Inserts the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("add")]
        public async Task<ActionResult> InsertAsync([FromBody] ReasonsModel model, [FromHeader] LocationHeader header)
        {
            model = (ReasonsModel)EmptyFilter.Handler(model);
            var response = await this.reasonsService.InsertAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Reasons,
                    LogDate = DateTime.Now,
                    LogFrom = (short)model.LoginRoleId,
                    LogDescription = $"<b>{model.CreatedByName}</b> has <b>added</b> the reason <b>{model.Reason}</b> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }
        /// <summary>
        /// Update the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("update")]
        public async Task<ActionResult> UpdateAsync([FromBody] ReasonsModel model, [FromHeader] LocationHeader header)
        {
            model = (ReasonsModel)EmptyFilter.Handler(model);
            var response = await this.reasonsService.UpdateAsync(model);
            if (response > 0)
            {
                var auditLog = new AuditLogModel
                {
                    AccountId = model.CreatedBy,
                    LogTypeId = (int)LogTypes.Reasons,
                    LogDate = DateTime.Now,
                    LogFrom = (short)model.LoginRoleId,
                    LogDescription = $"<b>{model.ModifiedByName}</b> has <b>updated</b> the reason <b>{model.Reason}</b> successfully.",
                    LocationId = Convert.ToInt32(header.LocationId)
                };
                await this.auditLogServices.LogAsync(auditLog);
            }

            return this.Success(response);
        }

        /// <summary>
        /// Fetches all asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("fetch-all")]
        public async Task<ActionResult> FetchAsync([FromBody] ReasonsModel model)
        {
            model = (ReasonsModel)EmptyFilter.Handler(model);
            var response = await this.reasonsService.FetchAllAsync(model);
            return this.Success(response);
        }
        /// <summary>
        /// Modify status the hoc asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The location.</param>
        /// <returns></returns>
        [HttpPost]
        [Route("modify-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ModifyStatusAsync([FromBody] ReasonsModel model, [FromHeader] LocationHeader header)
        {
            model = (ReasonsModel)EmptyFilter.Handler(model);
            var response = await this.reasonsService.ActivateOrDeactivateTest(model);
            if (response == 0)
            {
                return this.ServerError();
            }
            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                    LogTypeId = (int)LogTypes.Reasons,
                    LogDate = DateTime.Now,
                    LogFrom = short.Parse(model.LoginRoleId.ToString()),
                    LocationId = Convert.ToInt32(header.LocationId),
                    LogDescription = model.Active == true ? $@"<b>{model.ModifiedByName}</b> has <b>activated</b> the reason <b>{model.Reason}</b> on {DateTime.UtcNow.AddMinutes(330)}" : $@"<b>{model.ModifiedByName}</b> has <b>deactivated</b> the reason <b>{model.Reason}</b> on {DateTime.UtcNow.AddMinutes(330)}",

                };
    
        await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success(response);

        }
        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <param name="header">
        /// The location header.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        [HttpPost]
        [Route("delete")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] ReasonsModel model, [FromHeader] LocationHeader header)
        {
            try
            {
                model = (ReasonsModel)EmptyFilter.Handler(model);
                var response = await this.reasonsService.DeleteAsync(model);
                if (response == 0)
                {
                    return this.ServerError();
                }
                if (response > 0)
                {
                    var auditLog = new AuditLogModel
                    {
                        AccountId = model.CreatedBy,
                        LogTypeId = (int)LogTypes.Reasons,
                        LogDate = DateTime.Now,
                        LogFrom = (short)model.LoginRoleId,
                        LogDescription = $"<b>{model.ModifiedByName}</b> has <b>deleted</b> the reason <b>{model.Reason}</b> on {DateTime.UtcNow.AddMinutes(330)}",
                        LocationId = Convert.ToInt32(header.LocationId)
                    };
                    await this.auditLogServices.LogAsync(auditLog);
                }

                return this.Success("Reason has been deleted Successfully.");
            }
            catch (NpgsqlException exception)
            {
                if (exception.Message.Contains("violates foreign key constraint"))
                {
                    return this.Conflict("Reasons master can't be deleted. Please contact Administrator.");
                }
                return this.ServerError();
            }

        }
    
    }
}
